
/*****************************************************************************
 * 
 * Maze
 *
 * A TADS 3 class for processing a virtual maze
 *
 *****************************************************************************/

#include <adv3.h>
#include <en_us.h>

/*
 * Bit strings used to test if a bit is set in a cell of the maze
 *
 * (See the "mazeGrid" property)
 */

#define visited 0x10000

#define westwall 0x0008
#define southwall 0x0004
#define eastwall 0x0002
#define northwall 0x0001


class Maze : InitObject

    /*
     *   In your program, override the mazeGrid property with the Vector statement that
     *   was created by the JAVA component of the Virtual Maze Generator for TADS 3.
     *
     *   It describes every cell of the maze with this basic format:
     *
     *         h=1           h=2           h=3
     *   [[w=1,w=2,w=3],[w=1,w=2,w=3],[w=1,w=2,w=3]]
     *
     *   For each cell, the first 16 bits of its value contain the
     *   following information:
     *
     *   bit 1    unused
     *   bit 2    unused
     *   bit 3    unused
     *   bit 4    unused
     *   bit 5    unused
     *   bit 6    unused
     *   bit 7    unused
     *   bit 8    unused
     *   bit 9    unused
     *   bit 10   unused
     *   bit 11   unused
     *   bit 12   unused
     *   bit 13   unused
     *   bit 14   unused
     *   bit 15   unused
     *   bit 16   visited
     *
     *   For each cell, the last 16 bits of its value contain the
     *   following information:
     *
     *   bit 1    backtrack west
     *   bit 2    backtrack south
     *   bit 3    backtrack east
     *   bit 4    backtrack north
     *   bit 5    solution west
     *   bit 6    solution south
     *   bit 7    solution east
     *   bit 8    solution north
     *   bit 9    border west
     *   bit 10   border south
     *   bit 11   border east
     *   bit 12   border north
     *   bit 13   walls west
     *   bit 14   walls south
     *   bit 15   walls east
     *   bit 16   walls north
     *
     *   In version 1.0 of this class, only bits 13-16 are used.
     */
    mazeGrid = nil

    /*
     * Properties
     */

    /*
     *   randomStartFinish
     *
     *   true   you will have a random start position, and the maze will have
     *          a random exit position
     *
     *   nil    you will start at location 1,1
     *          the maze will have its exit in the last cell 
     */
    randomStartFinish = true
    
    mazeHeight = 0
    mazeWidth = 0
    StartCellh = 1
    StartCellw = 1
    CurrentCellh = 0
    CurrentCellw = 0
    FinishCellh = 0
    FinishCellw = 0
    
    /*
     *   atFinishCell
     *
     *   true   you are in the maze cell that contains the exit
     *
     *   nil    you are in a maze cell that does not contain the exit
     */
    atFinishCell = nil
    
    /*
     *   The Maze class's InitObject method.  It is run automatically when a 
     *   "Maze" object is created.
     */
    execute()
    {
        randomize();
        setStartCell();
        setFinishCell();
    }

    setVisited()
    {
        mazeGrid[CurrentCellh][CurrentCellw] = (mazeGrid[CurrentCellh][CurrentCellw] |
            visited);
    }
    
    setStartCell()
    {
        mazeHeight = mazeGrid.length();
        mazeWidth = mazeGrid[1].length();

        if (randomStartFinish)
        {
            StartCellh = rand(mazeHeight) + 1;
            StartCellw = rand(mazeWidth) + 1;
        }
        CurrentCellh = StartCellh;
        CurrentCellw = StartCellw;
        setVisited();
    }
    
    setFinishCell()
    {
        if (randomStartFinish)
        {
            do
            {
                FinishCellh = rand(mazeHeight) + 1;
                FinishCellw = rand(mazeWidth) + 1;
            }
            while ((FinishCellh == StartCellh) && (FinishCellw == StartCellw));
        }
        else
        {
            FinishCellh = mazeHeight;
            FinishCellw = mazeWidth;
        }
    }

    hasWestSet(h,w)
    {
        return ((mazeGrid[h][w] & westwall) == westwall);
    }

    hasSouthSet(h,w)
    {
        return ((mazeGrid[h][w] & southwall) == southwall);
    }

    hasEastSet(h,w)
    {
        return ((mazeGrid[h][w] & eastwall) == eastwall);
    }
    
    hasNorthSet(h,w)
    {
        return ((mazeGrid[h][w] & northwall) == northwall);
    }
        
/*********************************************************
 *
 * Process the maze crawler's movements
 *
 *********************************************************/

    bumpWallMessage()
    {
        "You are bumping into the wall\b";
    }
   
    /*
     *   This method is called after every successful move in the maze
     */
    isFinishCell()
    {
        if ((CurrentCellh == FinishCellh)&& (CurrentCellw == FinishCellw))
        {
            "Hooray!  You found the exit!";
            atFinishCell = true;
        }
        else
        {
            "You move one step.\b";
            atFinishCell = nil;
        }
        setVisited();
    }

    ProcessMaze(MazeValue)
    {
        switch (MazeValue)
        {
        case 'west':
            {
                /*
                 * The player wants to move west.
                 *
                 * If the current cell has a west wall, display an error message.
                 * Otherwise, change the current cell, and check if we are at the exit.
                 */
                if (hasWestSet(CurrentCellh,CurrentCellw))
                {
                    bumpWallMessage();
                }
                else
                {
                    CurrentCellw--;
                    isFinishCell();
                }
                break;
            }
        case 'south':
            {
                /*
                 * The player wants to move south.
                 *
                 * If the current cell has a south wall, display an error message.
                 * Otherwise, change the current cell, and check if we are at the exit.
                 */                
                if (hasSouthSet(CurrentCellh,CurrentCellw))
                {
                    bumpWallMessage();
                }
                else
                {
                    CurrentCellh++;
                    isFinishCell();
                }
                break;
            }            
        case 'east':
            {
                /*
                 * The player wants to move east.
                 *
                 * If the current cell has an east wall, display an error message.
                 * Otherwise, change the current cell, and check if we are at the exit.
                 */                
                if (hasEastSet(CurrentCellh,CurrentCellw))
                {
                    bumpWallMessage();
                }
                else
                {
                    CurrentCellw++;
                    isFinishCell();
                }
                break;
            }
        case 'north':
            {
                /*
                 * The player wants to move north.
                 *
                 * If the current cell has a north wall, display an error message.
                 * Otherwise, change the current cell, and check if we are at the exit.
                 */                
                if (hasNorthSet(CurrentCellh,CurrentCellw))
                {
                    bumpWallMessage();
                }
                else
                {
                    CurrentCellh--;
                    isFinishCell();
                }
                break;
            }        
        }
    }

    
/*********************************************************
 *
 * Display the Maze
 *
 * Notes about TADS 3
 *
 * 1. By default, TADS 3 uses a proportional font.  The space character in this font
 *    is narrower than the other characters; thus, it screws up the display of the
 *    map.
 *
 *    We must surround all output with the HTML tag <TT></TT> to use a
 *    fixed-width font.
 *
 * 2. TADS 3's output formatter automatically converts a string of spaces to a single
 *    space character, which screws up the display of the map.
 *
 *    Therefore, we must output all spaces as quoted spaces ('\ ') to force their
 *    display.
 *
 *********************************************************/

    hasVisited(h,w)
    {
        return ((mazeGrid[h][w] & visited) == visited);
    }

    DisplayMaze(entireMaze)
    {

        local h = 1;
        local w = 1;
        
        // For each row
        while (h <= mazeHeight)
        {
            // Go column by column from the left

            /*
             * We process each row four times.
             *
             * Pass 1: We print the north walls in the row.
             *
             * A cell is five characters in width.  However, for each
             * cell, we only print the first four characters because
             * adjacent cells share one character of the north wall.
             */
            while (w <= mazeWidth) {
               /*
                *   If the player wants to see the entire maze OR the player 
                *   wants to see a partial map and the cell has been visited
                */
               if (entireMaze || (!entireMaze && hasVisited(h,w)))
               {                  
                    if (hasNorthSet(h,w))
                    {
                        // Display the north wall
                        "<TT>====</TT>";
                    }
                    else
                    {
                        // Display a hole in the north wall
                        "<TT>==\ =</TT>";
                    }
                }
                else
                    /*
                     *   The player wants to see a partial map and the cell 
                     *   has not been visited.  Display nothing (spaces).
                     */
                    "<TT>\ \ \ \ </TT>";
                w++;
            }
            /*
             * For the last cell in each row, we print the final north wall character
             * and force a new line character to go to the next line of output.
             */

            /*
             *   If the player wants to see the entire maze OR the player 
             *   wants to see a partial map and the cell has been visited
             */ 
            if (entireMaze || (!entireMaze && hasVisited(h,w-1)))
                "<TT>=</TT>";
            else
                /*
                 *   The player wants to see a partial map and the cell 
                 *   has not been visited.  Display nothing (spaces).
                 */                
                "<TT>\ </TT>";
            "\n";

            /*
             * We process each row four times.
             *
             * Passes 2-4: We print the west and east walls in the row
             *
             * A cell is three characters in height.
             */
            
            // Three passes for each row
            for(local i=2; i<=4; i++) {
                w = 1;
                // Process a row
                while (w <= mazeWidth)
                {
                   /*
                    *   If the player wants to see the entire maze OR the player 
                    *   wants to see a partial map and the cell has been visited
                    */                    
                    if (entireMaze || (!entireMaze && hasVisited(h,w)))
                    {
                        // Pass 3: west wall or hole
                        if (i == 3)
                        {
                            if (hasWestSet(h,w))
                                //Display the west wall
                                "<TT># </TT>";
                            else
                                // Display a hole in the west wall
                                "<TT>\ \ </TT>";
                        }
                        else
                            // Pass 2 or 4: always wall
                            "<TT># </TT>";
                        /*
                         *
                         * The cell's interior
                         *
                         *   For pass 2, the top of the cell's interior
                         *
                         *       space     a normal cell
                         *       X         the maze's exit
                         *
                         *   For pass 3, the middle of the cell's interior
                         *
                         *       space     a normal cell
                         *       P         the player's current location
                         *
                         *   For pass 4, output a space for the bottom of the cell's interior. 
                         *
                         */
                        switch (i)                        
                        {
                        case 2:
                            if ((h == FinishCellh) && (w == FinishCellw))
                                "<TT><FONT COLOR=RED><B>X </B></FONT></TT>";
                            else
                                "<TT>\ \ </TT>";
                            break;
                        case 3:
                            if ((h == CurrentCellh) && (w == CurrentCellw))
                                "<TT><FONT COLOR=#99CC32><B>P </B></FONT></TT>";
                            else
                                "<TT>\ \ </TT>";
                            break;
                        case 4:
                            "<TT>\ \ </TT>";
                            break;
                        }                      
                    }
                    else
                        /*
                         *   The player wants to see a partial map and the cell 
                         *   has not been visited.  Display nothing (spaces).
                         */                        
                        "<TT>\ \ \ \ </TT>";
                    w++;
                }
                /*
                 * For the last cell in each row, we print the east wall character
                 * and force a new line character to go to the next line of output.
                 */
                
                /*
                 * If the player wants to see the entire maze OR the player 
                 * wants to see a partial map and the cell has been visited
                 */ 
                if (entireMaze || (!entireMaze && hasVisited(h,w-1)))
                    "<TT>#</TT>";
                else
                    /*
                     * The player wants to see a partial map and the cell 
                     * has not been visited.  Display nothing (spaces).
                     */
                    "<TT>\ </TT>";
                "\n";
            }

            w = 1;
            h++;
        }
        /*
         * Print the bottom border of the maze
         */
        for(local i=1; i<=mazeWidth; i++)
        {
            /*
             * If the player wants to see the entire maze OR the player 
             * wants to see a partial map and the cell has been visited
             */ 
            if (entireMaze || (!entireMaze && hasVisited(h-1,i)))
            {
                "<TT>====</TT>";
            }    
            else
                /*
                 *   The player wants to see a partial map and the cell 
                 *   has not been visited.  Display nothing (spaces).
                 */
                "<TT>\ \ \ \ </TT>";
        }
        
        /*
         *   Print the last character of the bottom border
         */
        
        /*
         * If the player wants to see the entire maze OR the player 
         * wants to see a partial map and the cell has been visited
         */
        if (entireMaze || (!entireMaze && hasVisited(h-1,mazeWidth)))
            "<TT>=</TT>";
        "\b";
        
        /*
         *   Print a legend
         */
        "<FONT COLOR=RED><B>X</B></FONT>\ \ \ \ \ Maze exit\b";
        "<FONT COLOR=#99CC32><B>P</B></FONT>\ \ \ \ \ Player's position\b";
         
    }        
    
;


/*********************************************************
 *
 *   Verbs for the Maze
 *
 *********************************************************/

function MazeOnlyVerb()
{
    "You can only use this command in the maze.\b";
    return;
}

/*
 * The player entered the 'mwest' command
 */
DefineIAction(mWest)
  execAction()
        {
        if (me.isIn(AnyRoom))
            {
            myMaze.ProcessMaze('west');
            }
        else
            {
            MazeOnlyVerb();
            }
        }
;

VerbRule(mWest)
  'mwest'
  : mWestAction
  verbPhrase = 'moving west'
;

/*************/

/*
 * The player entered the 'meast' command
 */
DefineIAction(mEast)
  execAction()
        {
        if (me.isIn(AnyRoom))
            {
            myMaze.ProcessMaze('east');
            }
        else
            {
            MazeOnlyVerb();
            }
        }
;

VerbRule(mEast)
  'meast'
  : mEastAction
  verbPhrase = 'moving east'
;

/*************/

/*
 * The player entered the 'mnorth' command
 */
DefineIAction(mNorth)
  execAction()
        {
        if (me.isIn(AnyRoom))
            {
            myMaze.ProcessMaze('north');
            }
        else
            {
            MazeOnlyVerb();
            }
        }
;

VerbRule(mNorth)
  'mnorth'
  : mNorthAction
  verbPhrase = 'moving north'
;

/*************/

/*
 * The player entered the 'msouth' command
 */
DefineIAction(mSouth)
  execAction()
        {
        if (me.isIn(AnyRoom))
            {
            myMaze.ProcessMaze('south');
            }
        else
            {
            MazeOnlyVerb();
            }
        }
;

VerbRule(mSouth)
  'msouth'
  : mSouthAction
  verbPhrase = 'moving south'
;

/*************/

/*
 * The player entered the 'mreveal' command
 */
DefineIAction(mReveal)
  execAction()
        {
        if (me.isIn(AnyRoom))
            {
            myMaze.DisplayMaze(true);
            }
        else
            {
            MazeOnlyVerb();
            }
        }
;

VerbRule(mReveal)
  'mreveal'
  : mRevealAction
  verbPhrase = 'revealing the entire maze'
;

/*************/

/*
 * The player entered the 'mvisited' command
 */
DefineIAction(mVisited)
  execAction()
        {
        if (me.isIn(AnyRoom))
            {
            myMaze.DisplayMaze(nil);
            }
        else
            {
            MazeOnlyVerb();
            }
        }
;

VerbRule(mVisited)
  'mvisited'
  : mVisitedAction
  verbPhrase = 'revealing the maze (visited areas only)'
;


